-- Domain kontrol sistemi veritabanı yapısı
CREATE DATABASE IF NOT EXISTS domain_tracker CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE domain_tracker;

CREATE TABLE IF NOT EXISTS domains (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_url VARCHAR(500) NOT NULL COMMENT 'Kontrol edilecek domain (örn: https://t.ly/vidoadres)',
    redirect_domain VARCHAR(500) DEFAULT NULL COMMENT 'Yönlendirilen domain',
    last_check DATETIME DEFAULT NULL COMMENT 'Son kontrol zamanı',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_domain (domain_url(255))
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Channels tablosu (Botun katıldığı kanal ve gruplar)
CREATE TABLE IF NOT EXISTS channels (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chat_id BIGINT NOT NULL COMMENT 'Telegram chat ID',
    chat_title VARCHAR(255) DEFAULT NULL COMMENT 'Kanal/Grup adı',
    chat_type VARCHAR(50) DEFAULT NULL COMMENT 'channel, group, supergroup',
    chat_username VARCHAR(255) DEFAULT NULL COMMENT 'Kanal username (@kanaladi)',
    is_active TINYINT(1) DEFAULT 0 COMMENT 'Aktif kanal mı? (mesaj gönderilecek)',
    first_seen DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT 'İlk görülme zamanı',
    last_activity DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_chat_id (chat_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Settings tablosu (Sistem ayarları)
CREATE TABLE IF NOT EXISTS settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE COMMENT 'Ayar anahtarı',
    setting_value TEXT DEFAULT NULL COMMENT 'Ayar değeri',
    description VARCHAR(255) DEFAULT NULL COMMENT 'Ayar açıklaması',
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Varsayılan ayarlar
INSERT INTO settings (setting_key, setting_value, description) VALUES
('active_channel_id', NULL, 'Mesaj gönderilecek aktif kanal ID'),
('admin_password', 'admin123', 'Admin paneli şifresi (değiştirin!)'),
('bot_token', NULL, 'Telegram bot token'),
('tly_api_key', NULL, 't.ly API anahtarı')
ON DUPLICATE KEY UPDATE setting_key=setting_key;

-- Domain History tablosu (Domain değişiklik geçmişi)
CREATE TABLE IF NOT EXISTS domain_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_id INT NOT NULL COMMENT 'Domain ID',
    domain_url VARCHAR(500) NOT NULL COMMENT 'Kontrol edilen domain URL',
    old_redirect_domain VARCHAR(500) DEFAULT NULL COMMENT 'Eski yönlendirilen domain',
    new_redirect_domain VARCHAR(500) NOT NULL COMMENT 'Yeni yönlendirilen domain',
    check_status VARCHAR(50) DEFAULT 'success' COMMENT 'success, error',
    error_message TEXT DEFAULT NULL COMMENT 'Hata mesajı (varsa)',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_domain_id (domain_id),
    INDEX idx_created_at (created_at),
    FOREIGN KEY (domain_id) REFERENCES domains(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Failed Messages tablosu (Başarısız Telegram mesajları)
CREATE TABLE IF NOT EXISTS failed_messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_id INT NOT NULL COMMENT 'Domain ID',
    domain_url VARCHAR(500) NOT NULL COMMENT 'Domain URL',
    redirect_domain VARCHAR(500) NOT NULL COMMENT 'Yönlendirilen domain',
    channel_id BIGINT NOT NULL COMMENT 'Telegram channel ID',
    message_text TEXT NOT NULL COMMENT 'Gönderilecek mesaj',
    keyboard_data TEXT DEFAULT NULL COMMENT 'Butonlar (JSON)',
    error_message TEXT DEFAULT NULL COMMENT 'Hata mesajı',
    retry_count INT DEFAULT 0 COMMENT 'Tekrar deneme sayısı',
    max_retries INT DEFAULT 3 COMMENT 'Maksimum tekrar deneme sayısı',
    status VARCHAR(50) DEFAULT 'pending' COMMENT 'pending, retrying, failed, sent',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    last_retry_at DATETIME DEFAULT NULL,
    INDEX idx_status (status),
    INDEX idx_retry_count (retry_count),
    INDEX idx_created_at (created_at),
    FOREIGN KEY (domain_id) REFERENCES domains(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Örnek domain ekleme
INSERT INTO domains (domain_url) VALUES ('https://t.ly/vidoadres') 
ON DUPLICATE KEY UPDATE domain_url=domain_url;

